import os
import sys
import subprocess

# Ensure 'code' directory is in path
sys.path.append(os.path.join(os.path.dirname(__file__), 'code'))

def run_step(script_name, description):
    print('')
    print('=' * 60)
    print(f'STEP: {description}')
    print(f'RUNNING: {script_name}')
    print('=' * 60)

    # Check locations: Try root first, then 'code' subdirectory
    if os.path.exists(script_name):
        script_path = script_name
    elif os.path.exists(os.path.join('code', script_name)):
        script_path = os.path.join('code', script_name)
    else:
        print(f'ERROR: Could not find {script_name} in root or code/ directory.')
        sys.exit(1)
    
    try:
        subprocess.run([sys.executable, script_path], check=True)
        print(f'SUCCESS: {script_name} completed.')
    except subprocess.CalledProcessError:
        print(f'ERROR: Failed to run {script_name}')
        sys.exit(1)

if __name__ == '__main__':
    print('--- REFRACTIVE COSMOLOGY PIPELINE (v2.0) ---')

    # 1. Data Generation & Rectification
    # Saves to produced/refined_actual_data_1701.csv and produced/model_predictions_1701.csv
    run_step('generate_clean_v0_data.py', 'Data Generation: Rectification & K-Convergence')

    # 2. Primary Results Plotting
    # Saves to plots/velocity_potential_fit.png
    run_step('plot_results.py', 'Plotting: Velocity Potential & Saturation')

    # 3. Statistical Robustness
    # Saves to plots/robustness_comparison_final.png
    run_step('run_refractive_analysis.py', 'Analysis: Statistical Robustness Check')

    # 4. Time Dilation
    # Saves to produced/appendix_D_time_dilation.txt
    run_step('calc_time_dilation.py', 'Physics: Time Dilation & Horizon Topology')

    # 5. The Bridge Plot
    # Saves to plots/hubble_residuals_bridge.png
    run_step('hubble_residuals.py', 'Plotting: Origin of the Dark Energy Signal')

    print('')
    print('=' * 60)
    print('--- PIPELINE COMPLETE ---')
    print("Data artifacts: produced/")
    print("Figure artifacts: plots/")
    print('=' * 60)